﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Runtime.InteropServices;
using System.ServiceProcess;
using System.Text;
using System.Threading.Tasks;

namespace LogoffService
{
    public partial class LogoffService : ServiceBase
    {
        private Dictionary<int, string> loginid = new Dictionary<int, string>();

        [DllImport("wtsapi32.dll")]
        public static extern bool WTSEnumerateSessions(
            IntPtr hServer,
            uint reserved,
            uint version,
            out IntPtr ppSessionInfo,
            out uint pCount);

        [DllImport("wtsapi32.dll")]
        public static extern bool WTSEnumerateSessionsW(
                    IntPtr hServer,
                    uint reserved,
                    uint version,
                    out IntPtr ppSessionInfo,
                    out uint pCount);


        [DllImport("wtsapi32.dll")]
        public static extern bool WTSQuerySessionInformation(
            IntPtr hServer,
            uint sessionId,
            WTS_INFO_CLASS wtsInfoClass,
            out IntPtr ppBuffer,
            out uint iBytesReturned);

        [DllImport("wtsapi32.dll")]
        static extern void WTSFreeMemory(IntPtr pMemory);

        [DllImport(@"Kernel32.dll")]
        public extern static UInt32 WTSGetActiveConsoleSessionId();


        public enum WTS_INFO_CLASS
        {
            WTSInitialProgram,
            WTSApplicationName,
            WTSWorkingDirectory,
            WTSOEMId,
            WTSSessionId,
            WTSUserName,
            WTSWinStationName,
            WTSDomainName,
            WTSConnectState,
            WTSClientBuildNumber,
            WTSClientName,
            WTSClientDirectory,
            WTSClientProductId,
            WTSClientHardwareId,
            WTSClientAddress,
            WTSClientDisplay,
            WTSClientProtocolType,
            WTSIdleTime,
            WTSLogonTime,
            WTSIncomingBytes,
            WTSOutgoingBytes,
            WTSIncomingFrames,
            WTSOutgoingFrames,
            WTSClientInfo,
            WTSSessionInfo,
            WTSConfigInfo,
            WTSValidationInfo,
            WTSSessionAddressV4,
            WTSIsRemoteSession
        }

        [StructLayout(LayoutKind.Sequential)]
        private struct WTS_SESSION_INFO
        {
            public uint SessionID;
            [MarshalAs(UnmanagedType.LPStr)]
            public string WinStationName;
            public WTS_CONNECTSTATE_CLASS State;
        }

        public enum WTS_CONNECTSTATE_CLASS
        {
            WTSActive,
            WTSConnected,
            WTSConnectQuery,
            WTSShadow,
            WTSDisconnected,
            WTSIdle,
            WTSListen,
            WTSReset,
            WTSDown,
            WTSInit
        }


        public LogoffService()
        {
            InitializeComponent();
        }

        protected override void OnStart(string[] args)
        {
            base.OnStart(args);

            // サービス開始時にアクティブユーザのセッションIDとログインIDを取得
            Dictionary<int, string> loginIdOnStart = GetLoginIdActiveSession();
            // ログインIDが取得できたとき
            if (loginIdOnStart != null)
            {
                
                // キーを取得しキーに紐づくログインIDが存在しないとき処理を抜ける
                var keysOnStart = loginIdOnStart.Keys;

                if (keysOnStart.Count == 0 || loginIdOnStart[keysOnStart.First<int>()] == "")
                {
                    return;
                }
                // セッションIDとログインIDのペアが存在するとき
                loginid = loginIdOnStart;
                var keys = loginid.Keys;
                if(keys.Count != 0)
                {
                    string str = "start" + "," + loginid[keys.First<int>()] + "," + keys.First<int>().ToString() + "," + DateTime.Now.ToString("yyyy/MM/dd HH:mm:ss");
                    writelog(str);
                }

            }
        }

        protected override void OnStop()
        {
        }

        protected override void OnSessionChange(SessionChangeDescription changeDescription)
        {
            base.OnSessionChange(changeDescription);

            if (changeDescription.Reason == SessionChangeReason.SessionLogon)
            {
                // ログイン時にログインIDを取得
                string tmp = GetLoginId(changeDescription.SessionId);
                if (loginid.ContainsKey(changeDescription.SessionId))
                {
                    loginid[changeDescription.SessionId] = tmp;
                }
                else
                {
                    loginid.Add(changeDescription.SessionId, tmp);
                }
                if (tmp != "")
                {
                    string str = "logon" + "," + loginid[changeDescription.SessionId] + "," + changeDescription.SessionId.ToString() + "," + DateTime.Now.ToString("yyyy/MM/dd HH:mm:ss");
                    writelog(str);
                }
            }
            else if (changeDescription.Reason == SessionChangeReason.SessionLogoff)
            {
                // ログオフ時に取得したログインIDを利用
                // ここではログインIDをとれないため
                string tmp = loginid[changeDescription.SessionId];

                if (tmp != "")
                {
                    string str = "logoff" + "," + loginid[changeDescription.SessionId] + "," + changeDescription.SessionId.ToString() + "," + DateTime.Now.ToString("yyyy/MM/dd HH:mm:ss");
                    writelog(str);
                }
            }

        }

        private void writelog(string line)
        {
            if (!Directory.Exists(@"C:\LogoffLog"))
            {
                Directory.CreateDirectory(@"C:\LogoffLog");
            }

            string file = DateTime.Now.ToString("yyyyMMddHHmmss") + ".txt";

            string path = Path.Combine(@"C:\LogoffLog", file);
            using (var fs = new StreamWriter(path))
            {
                fs.WriteLine(line);
            }

        }

        // セッションIDに紐づくログインIDを取得する
        public string GetLoginId(int pSessionId)
        {
            var WTS_CURRENT_SERVER_HANDLE = IntPtr.Zero;
            IntPtr ppSessionInfo;
            uint pCount;
            if (!WTSEnumerateSessionsW(
                WTS_CURRENT_SERVER_HANDLE,
                0,
                1,
                out ppSessionInfo,
                out pCount))
            {
                return "";
            }

            var iDataSize = Marshal.SizeOf(typeof(WTS_SESSION_INFO));
            var current = new IntPtr(ppSessionInfo.ToInt32());
            for (var i = 0; i < pCount; i++)
            {
                var sessionInfo = (WTS_SESSION_INFO)Marshal.PtrToStructure(current, typeof(WTS_SESSION_INFO));
                current = current + iDataSize;

                var sessionId = sessionInfo.SessionID;

                IntPtr ppBuffer;
                uint iReturned;

                if (WTSQuerySessionInformation(WTS_CURRENT_SERVER_HANDLE,
                    sessionId,
                    WTS_INFO_CLASS.WTSUserName,
                    out ppBuffer,
                    out iReturned))
                {
                    //var dwSesId = WTSGetActiveConsoleSessionId();
                    if (sessionId == pSessionId)
                    {
                        string loginId = Marshal.PtrToStringAnsi(ppBuffer);
                        WTSFreeMemory(ppBuffer);
                        WTSFreeMemory(ppSessionInfo);
                        return loginId;
                    }
                    else
                    {
                        WTSFreeMemory(ppBuffer);
                    }
                }
            }


            WTSFreeMemory(ppSessionInfo);
            return "";

        }

        // アクティブなセッションIDおよびログインIDを取得する
        public Dictionary<int, string> GetLoginIdActiveSession()
        {
            var WTS_CURRENT_SERVER_HANDLE = IntPtr.Zero;
            IntPtr ppSessionInfo;
            uint pCount;
            if (!WTSEnumerateSessionsW(
                WTS_CURRENT_SERVER_HANDLE,
                0,
                1,
                out ppSessionInfo,
                out pCount))
            {
                return null;
            }

            var iDataSize = Marshal.SizeOf(typeof(WTS_SESSION_INFO));
            var current = new IntPtr(ppSessionInfo.ToInt32());
            for (var i = 0; i < pCount; i++)
            {
                var sessionInfo = (WTS_SESSION_INFO)Marshal.PtrToStructure(current, typeof(WTS_SESSION_INFO));
                current = current + iDataSize;

                var sessionId = sessionInfo.SessionID;

                IntPtr ppBuffer;
                uint iReturned;

                if (WTSQuerySessionInformation(WTS_CURRENT_SERVER_HANDLE,
                    sessionId,
                    WTS_INFO_CLASS.WTSUserName,
                    out ppBuffer,
                    out iReturned))
                {
                    var dwSesId = WTSGetActiveConsoleSessionId();
                    if (sessionId == dwSesId)
                    {
                        Dictionary<int, string> ret = new Dictionary<int, string>();
                        string loginId = Marshal.PtrToStringAnsi(ppBuffer);
                        ret.Add((int)sessionId, loginId);
                        WTSFreeMemory(ppBuffer);
                        WTSFreeMemory(ppSessionInfo);
                        return ret;
                    }
                    else
                    {
                        WTSFreeMemory(ppBuffer);
                    }
                }
            }


            WTSFreeMemory(ppSessionInfo);
            return null;



        }
    }
}
